<?php
/**
 * @brief Freelancer, a theme for Dotclear 2
 *
 * @package Dotclear
 * @subpackage Themes
 *
 * @author Philippe aka amalgame and contributors
 * @copyright GPL-2.0
 */
declare(strict_types=1);

namespace Dotclear\Theme\freelancer;

use Dotclear\App;
use Dotclear\Core\Process;
use Dotclear\Core\Backend\Page;
use Dotclear\Core\Backend\Notices;
use Exception;
use form;

class Config extends Process
{
    public static function init(): bool
    {
        // limit to backend permissions
        if (!self::status(My::checkContext(My::CONFIG))) {
            return false;
        }

        My::l10n('admin');

        App::backend()->standalone_config = (bool) App::themes()->moduleInfo(App::blog()->settings->system->theme, 'standalone_config');

        // Load contextual help
        App::themes()->loadModuleL10Nresources(My::id(), App::lang()->getLang());

        App::backend()->freelancer_default_image_url = My::fileURL('/img/home.png');

        $style = App::blog()->settings->themes->get(App::blog()->settings->system->theme . '_style');
        $style = $style ? (unserialize($style) ?: []) : [];

        if (!is_array($style)) {
            $style = [];
        }

        if (!isset($style['freelancer_user_image']) || empty($style['freelancer_user_image'])) {
            $style['freelancer_user_image'] = App::backend()->freelancer_default_image_url;
        }

        App::backend()->style     = $style;
        App::backend()->theme_url = My::fileURL('');

        return self::status();
    }

    /**
     * Processes the request(s).
     */
    public static function process(): bool
    {
        if (!self::status()) {
            return false;
        }

        if (!empty($_POST)) {
            try {
                // HTML
                $style = [];
                if (!empty($_POST['freelancer_user_image'])) {
                    $style['freelancer_user_image'] = $_POST['freelancer_user_image'];
                } else {
                    $style['freelancer_user_image'] = App::backend()->freelancer_default_image_url;
                }

                App::backend()->style = $style;

                App::blog()->settings->themes->put(App::blog()->settings->system->theme . '_style', serialize(App::backend()->style));

                // Blog refresh
                App::blog()->triggerBlog();

                // Template cache reset
                App::cache()->emptyTemplatesCache();

                Notices::message(__('Theme configuration upgraded.'), true, true);
            } catch (Exception $e) {
                App::error()->add($e->getMessage());
            }
        }

        return true;
    }

    /**
     * Renders the page.
     */
    public static function render(): void
    {
        if (!self::status()) {
            return;
        }

        if (!App::backend()->standalone_config) {
            echo '</form>';
        }

        echo '<form id="theme_config" action="' . App::backend()->url()->get('admin.blog.theme', ['conf' => '1']) .
        '" method="post" enctype="multipart/form-data">';

        echo '<h4 class="pretty-title">' . __('Profile image') . '</h4>';

        echo '<div class="box theme">';

        echo '<p> ' .
        '<img id="freelancer_user_image_src" alt="' . __('Thumbnail') . '" src="' . App::backend()->style['freelancer_user_image'] . '" class="img-profile" />' .
         '</p>';

        echo '<p class="freelancer-buttons"><button type="button" id="freelancer_user_image_selector">' . __('Change') . '</button>' .
        '<button class="delete" type="button" id="freelancer_user_image_reset">' . __('Reset') . '</button>' .
        '</p>' ;

        echo '<p class="hidden-if-js">' . form::field('freelancer_user_image', 30, 255, App::backend()->style['freelancer_user_image']) . '</p>';

        echo '</div>';

        echo '<p class="clear"><input type="submit" value="' . __('Save') . '" />' . App::nonce()->getFormNonce() . '</p>';
        echo form::hidden(['theme-url'], App::backend()->theme_url);
        echo '</form>';

        Page::helpBlock('freelancer');

        // Legacy mode
        if (!App::backend()->standalone_config) {
            echo '<form style="display:none">';
        }
    }
}
